<?php

namespace StInMaWi\Posts;

use PDO;
use PDOException;

/** @package StInMaWi\Posts */
class PostsRepository
{
    /**
     * @param PDO $pdo 
     * @return void 
     */
    public function __construct(protected PDO $pdo) { }

    /**
     * @return array 
     * @throws PDOException 
     */
    public function getPosts(): array
    {
        return $this -> getAll();
    }

    /**
     * @return array 
     * @throws PDOException 
     */
    public function getAll(): array
    {
        $stmt = $this -> pdo -> prepare('SELECT `id`, `slug`, `title`, `content` FROM `posts` ORDER BY `id` ASC');
        $stmt -> execute();
        return $stmt -> fetchAll(PDO::FETCH_CLASS, PostsModel::class);
    }

    /**
     * @param string $slug 
     * @return null|PostsModel 
     * @throws PDOException 
     */
    public function getPost(string $slug): ?PostsModel
    {
        $stmt = $this -> pdo -> prepare('SELECT `id`, `slug`, `title`, `content` FROM `posts` WHERE `slug` = :slug');
        $stmt -> bindValue(':slug', $slug);
        $stmt -> setFetchMode(PDO::FETCH_CLASS, PostsModel::class);
        $stmt -> execute();
        $entry = $stmt -> fetch();

        if(!empty($entry))
        {
            return $entry;
        }
        else
        {
            return null;
        }
    }

    public function findById(int $id): ?PostsModel
    {
        $stmt = $this->pdo->prepare('SELECT `id`, `slug`, `title`, `content` FROM `posts` WHERE `id` = :id');
        $stmt -> bindValue(':id', $id);
        $stmt -> setFetchMode(PDO::FETCH_CLASS, PostsModel::class);
        $stmt -> execute();
        $entry = $stmt -> fetch();

        if(!empty($entry))
        {
            return $entry;
        }
        else
        {
            return null;
        }
    }
    
    public function createPost(string $title, string $slug, string $content): bool
    {
        $existsStmt = $this -> pdo -> prepare('SELECT COUNT(*) AS c FROM `posts` WHERE `slug` = :slug');
        $existsStmt -> bindValue(':slug', $slug);
        $existsStmt -> setFetchMode(PDO::FETCH_ASSOC);
        $existsStmt -> execute();
        $existsValue = $existsStmt -> fetch();
        if (empty($existsValue) || $existsValue['c'] != 0) {
            return false;
        }

        $stmt = $this -> pdo -> prepare('INSERT INTO `posts` (title, slug, content) VALUES (:title, :slug, :content)');
        $stmt -> bindValue(':title', $title);
        $stmt -> bindValue(':slug', $slug);
        $stmt -> bindValue(':content', $content);
        $stmt -> execute();

        return true;
    }

    public function updatePost(int $id, string $title, string $content)
    {
        $stmt = $this -> pdo -> prepare('UPDATE `posts` SET `title` = :title, `content` = :content WHERE `id` = :id ');
        $stmt -> bindValue(':title', $title);
        $stmt -> bindValue(':content', $content);
        $stmt -> bindValue(':id', $id);
        $stmt -> execute();
    }

    public function deletePost($id)
    {
        $stmt = $this -> pdo -> prepare('DELETE FROM `posts` WHERE `id` = :id');
        $stmt -> bindValue(':id', $id);
        $stmt -> execute();
    }
}